document.addEventListener('DOMContentLoaded', () => {
    console.log("Massive Events Website Loaded");

    // Check for success parameter in URL (Thank You Modal)
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('success') === 'true') {
        const modal = document.getElementById('thankYouModal');
        if (modal) {
            modal.style.display = 'flex';

            // Clean up URL without refreshing
            const newUrl = window.location.pathname;
            window.history.replaceState({}, document.title, newUrl);

            // Close logic
            const closeBtn = modal.querySelector('.close-modal');
            const okBtn = modal.querySelector('.ok-btn');

            const closeModal = () => {
                modal.style.display = 'none';
            };

            if (closeBtn) closeBtn.addEventListener('click', closeModal);
            if (okBtn) okBtn.addEventListener('click', closeModal);

            // Close on outside click
            window.addEventListener('click', (e) => {
                if (e.target === modal) {
                    closeModal();
                }
            });
        }
    }

    // Mobile Menu Toggle
    const menuToggle = document.querySelector('.menu-toggle');
    const navMenu = document.querySelector('.nav-menu');

    if (menuToggle) {
        menuToggle.addEventListener('click', () => {
            navMenu.classList.toggle('active');
            menuToggle.classList.toggle('active'); // detailed animation if needed
        });
    }

    // Lightbox Functionality
    const masonryItems = document.querySelectorAll('.masonry-item');
    if (masonryItems.length > 0) {
        const lightbox = document.getElementById('lightbox');
        const lightboxImg = document.getElementById('lightbox-img');
        const closeBtn = document.querySelector('.close-lightbox');
        const prevBtn = document.querySelector('.prev');
        const nextBtn = document.querySelector('.next');

        let currentIndex = 0;
        const images = Array.from(document.querySelectorAll('.gallery-img'));
        const imageUrls = images.map(img => img.src);

        const openLightbox = (index) => {
            lightbox.style.display = "block";
            lightboxImg.src = imageUrls[index];
            currentIndex = index;
            document.body.style.overflow = 'hidden'; // Disable scroll
        }

        const closeLightboxModal = () => {
            lightbox.style.display = "none";
            document.body.style.overflow = 'auto'; // Enable scroll
        }

        const showNext = () => {
            currentIndex = (currentIndex + 1) % imageUrls.length;
            lightboxImg.src = imageUrls[currentIndex];
        }

        const showPrev = () => {
            currentIndex = (currentIndex - 1 + imageUrls.length) % imageUrls.length;
            lightboxImg.src = imageUrls[currentIndex];
        }

        // Click on Item
        masonryItems.forEach((item, index) => {
            item.addEventListener('click', () => openLightbox(index));
        });

        // Close
        closeBtn.addEventListener('click', closeLightboxModal);

        // Navigation
        nextBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            showNext();
        });

        prevBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            showPrev();
        });

        // Close on clicking outside image
        lightbox.addEventListener('click', (e) => {
            if (e.target === lightbox) {
                closeLightboxModal();
            }
        });

        // Keyboard support
        document.addEventListener('keydown', (e) => {
            if (lightbox.style.display === "block") {
                if (e.key === "Escape") closeLightboxModal();
                if (e.key === "ArrowRight") showNext();
                if (e.key === "ArrowLeft") showPrev();
            }
        });
    }
    // Testimonial Carousel
    const track = document.querySelector('.testimonial-track');
    if (track) {
        const slides = Array.from(track.children);
        const nextButton = document.querySelector('.next-btn');
        const prevButton = document.querySelector('.prev-btn');
        const dotsNav = document.querySelector('.carousel-nav');
        const dots = Array.from(dotsNav.children);

        // Function to set slide positions
        const setSlidePosition = (slide, index) => {
            // Using 100% width relative to container
            slide.style.left = (index * 100) + '%';
        };
        slides.forEach(setSlidePosition);

        const moveToSlide = (track, currentSlide, targetSlide) => {
            track.style.transform = 'translateX(-' + targetSlide.style.left + ')';
            currentSlide.classList.remove('current-slide');
            targetSlide.classList.add('current-slide');
        }

        const updateDots = (currentDot, targetDot) => {
            currentDot.classList.remove('current-slide');
            targetDot.classList.add('current-slide');
        }

        // Next Button
        nextButton.addEventListener('click', e => {
            const currentSlide = track.querySelector('.current-slide');
            const nextSlide = currentSlide.nextElementSibling || slides[0];
            const currentDot = dotsNav.querySelector('.current-slide');
            const nextDot = currentDot.nextElementSibling || dots[0];

            moveToSlide(track, currentSlide, nextSlide);
            updateDots(currentDot, nextDot);
        });

        // Prev Button
        prevButton.addEventListener('click', e => {
            const currentSlide = track.querySelector('.current-slide');
            const prevSlide = currentSlide.previousElementSibling || slides[slides.length - 1];
            const currentDot = dotsNav.querySelector('.current-slide');
            const prevDot = currentDot.previousElementSibling || dots[dots.length - 1];

            moveToSlide(track, currentSlide, prevSlide);
            updateDots(currentDot, prevDot);
        });

        // Dot Nav
        dotsNav.addEventListener('click', e => {
            const targetDot = e.target.closest('button');
            if (!targetDot) return;

            const currentSlide = track.querySelector('.current-slide');
            const currentDot = dotsNav.querySelector('.current-slide');
            const targetIndex = dots.findIndex(dot => dot === targetDot);
            const targetSlide = slides[targetIndex];

            moveToSlide(track, currentSlide, targetSlide);
            updateDots(currentDot, targetDot);
        });

        // Auto-play (2 seconds)
        let slideInterval = setInterval(() => {
            nextButton.click();
        }, 2000);

        // Pause on hover
        const carouselContainer = document.querySelector('.testimonial-carousel-container');
        carouselContainer.addEventListener('mouseenter', () => {
            clearInterval(slideInterval);
        });

        carouselContainer.addEventListener('mouseleave', () => {
            slideInterval = setInterval(() => {
                nextButton.click();
            }, 2000);
        });
    }
    // Form Validation
    const contactForm = document.querySelector('form');
    if (contactForm) {
        const inputs = contactForm.querySelectorAll('input, textarea');

        const validateField = (field) => {
            const errorSpan = field.nextElementSibling;
            let isValid = false;

            // Check if user has left the field empty (only on blur)
            if (field.validity.valueMissing) {
                // If it's a date field, the browser uses a special UI so sometimes valueMissing checks differently, but standard works
                // Just use the default message or our custom one?
                // We'll rely on the pattern check mostly
            }

            // Custom Regex Checks (Fallback for pattern mismatch not verifying nicely in all browsers immediately)
            if (field.id === 'phone') {
                // Force numbers only (remove non-digits if typing? No, just validate)
                if (!/^[0-9]+$/.test(field.value) && field.value !== '') {
                    field.setCustomValidity("Invalid");
                } else {
                    field.setCustomValidity("");
                }
            }

            if (field.id === 'name') {
                if (!/^[a-zA-Z\s]+$/.test(field.value) && field.value !== '') {
                    field.setCustomValidity("Invalid");
                } else {
                    field.setCustomValidity("");
                }
            }

            if (field.checkValidity()) {
                field.classList.remove('invalid');
                field.classList.add('valid');
                if (errorSpan && errorSpan.classList.contains('error-message')) {
                    errorSpan.classList.remove('visible');
                }
                isValid = true;
            } else {
                field.classList.remove('valid');
                field.classList.add('invalid');
                if (errorSpan && errorSpan.classList.contains('error-message')) {
                    errorSpan.classList.add('visible');
                }
                isValid = false;
            }
            return isValid;
        };

        inputs.forEach(input => {
            // Validate on blur (tab out)
            input.addEventListener('blur', () => {
                validateField(input);
            });

            // Re-validate on input (clear error if fixed)
            input.addEventListener('input', () => {
                if (input.classList.contains('invalid')) {
                    validateField(input);
                }
                // Special handling for Phone to restrict input to numbers only in real-time
                if (input.id === 'phone') {
                    input.value = input.value.replace(/[^0-9]/g, '');
                }
            });
        });

        // Handle submission
        contactForm.addEventListener('submit', (e) => {
            e.preventDefault();

            let formValid = true;
            inputs.forEach(input => {
                if (!validateField(input)) {
                    formValid = false;
                }
            });

            if (formValid) {
                const submitBtn = contactForm.querySelector('button[type="submit"]');
                const originalBtnText = submitBtn.innerText;
                const statusDiv = contactForm.querySelector('.form-status');

                // Disable button and show loading state
                submitBtn.disabled = true;
                submitBtn.innerText = 'Sending...';
                if (statusDiv) statusDiv.style.display = 'none';

                // Collect data
                const formData = new FormData(contactForm);
                const data = Object.fromEntries(formData.entries());

                // Add timestamp (client-side backup, though script handles it too)
                data.timestamp = new Date().toISOString();

                // YOUR GOOGLE APPS SCRIPT WEB APP URL GOES HERE
                // Example: const GOOGLE_SCRIPT_URL = 'https://script.google.com/macros/s/AKfycbx.../exec';
                const GOOGLE_SCRIPT_URL = 'https://script.google.com/macros/s/AKfycbwk1qk8TYXx_ZT_yDKSBSY0pcaHEIy9rwUrDZcrOl_dKQ0XhY211z4MbU1JgVsOp4DcFQ/exec';

                if (GOOGLE_SCRIPT_URL === 'https://script.google.com/macros/s/AKfycbwk1qk8TYXx_ZT_yDKSBSY0pcaHEIy9rwUrDZcrOl_dKQ0XhY211z4MbU1JgVsOp4DcFQ/exec') {
                    alert('Please configure the Google Script URL in js/main.js');
                    submitBtn.disabled = false;
                    submitBtn.innerText = originalBtnText;
                    return;
                }

                // Send data using fetch
                fetch(GOOGLE_SCRIPT_URL, {
                    method: 'POST',
                    mode: 'no-cors', // Important for Google Apps Script
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                })
                    .then(response => {
                        // With no-cors, we get an opaque response, so we assume success if no error thrown
                        // Reset form
                        contactForm.reset();
                        submitBtn.innerText = 'Sent!';

                        // Show Thank You Modal
                        const modal = document.getElementById('thankYouModal');
                        if (modal) {
                            modal.style.display = 'flex';
                            // Ensure close logic is attached (it is at top of this file)
                        }

                        setTimeout(() => {
                            submitBtn.disabled = false;
                            submitBtn.innerText = originalBtnText;
                        }, 2000);
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        if (statusDiv) {
                            statusDiv.innerText = "Something went wrong. Please try again or contact us directly.";
                            statusDiv.style.color = "red";
                            statusDiv.style.display = "block";
                        } else {
                            alert("There was an error sending your message. Please try again.");
                        }
                        submitBtn.disabled = false;
                        submitBtn.innerText = originalBtnText;
                    });
            }
        });
    }
});
